/*
 * CollisionActor.h
 *
 * Created 8/17/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/17/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef COLLISION_ACTOR_H
 #define COLLISION_ACTOR_H
 
 template <typename T> class CollisionActor;
 
 #include "Actor.h"
 #include "CollisionObject.h"
 
 #include "global.h"
 
 /**
  * Class specification for CollisionActor
  */
 template <typename T>
 class CollisionActor : virtual public Actor<T>, virtual public CollisionObject<T>
 {
 // Data Members
 private:
 
 // Local Functions
 public:
    CollisionActor();
    CollisionActor( const CollisionActor<T>& col_actor );
    virtual ~CollisionActor();
    inline CollisionActor<T>& operator=( const CollisionActor<T>& col_actor );
    virtual inline std::string get_default_collider_name() const = 0;
    inline void loop( const std::string& anim_name );
    inline void play( const std::string& anim_name );
    inline void pose( const std::string& anim_name, const int frame );
    inline void stop();
    inline void stop( const std::string& anim_name );
    virtual inline void turn_on_collision();
    virtual inline void turn_off_collision();
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "CollisionActor<" + template_type + ">" );
                            }*/
 
 // Private Functions
 private:
    
 // Friend Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 CollisionActor<T>::CollisionActor()
                   : Actor<T>(),
                     CollisionObject<T>()
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CollisionActor<T>::CollisionActor( const CollisionActor<T>& col_actor )
                   : Actor<T>( col_actor ),
                     CollisionObject<T>( col_actor )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CollisionActor<T>::~CollisionActor()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified CollisionActor to this CollisionActor.
  *
  * @param (const CollisionActor<T>& col_actor )
  * @return CollisionActor<T>&
  */
 template <typename T>
 inline CollisionActor<T>& CollisionActor<T>::operator=( const CollisionActor<T>& col_actor )
 {
    Actor<T>::operator=( static_cast< Actor<T> >( col_actor ) );
    CollisionObject<T>::operator=( static_cast< CollisionObject<T> >( col_actor ) );
    
    return *this;
 }
 
 /**
  * loop() changes the animation and collider to the animation and collider 
  * having the specified anim_name. The animation is looped.
  *
  * @param (const std::string&) anim_name
  */
 template <typename T>
 inline void CollisionActor<T>::loop( const std::string& anim_name )
 {
    AnimControl* anim_Ptr( Actor<T>::get_anims().find_anim( anim_name ) );
    
    if ( anim_Ptr != NULL && !anim_Ptr->is_playing() )
    {
        // stop current animation and detach current collider
        CollisionActor<T>::stop();
        
        // change the animation
        anim_Ptr->loop( true );
        
        // change the collider
        CollisionObject<T>::reparent_collider_to( anim_name, *this );
    }
 }
 
 /**
  * play() changes the animation and collider to the animation and collider 
  * having the specified anim_name. The animation is only played once.
  *
  * @param (const std::string&) anim_name 
  */
 template <typename T>
 inline void CollisionActor<T>::play( const std::string& anim_name )
 {
    AnimControl* anim_Ptr( Actor<T>::get_anims().find_anim( anim_name ) );
    
    if ( anim_Ptr != NULL && !anim_Ptr->is_playing() )
    {
        // stop current animation and detach current collider
        CollisionActor<T>::stop();
        
        // change the animation
        anim_Ptr->play();
        
        // change the collider
        CollisionObject<T>::get_colliders().reparent_to( anim_name, *this );
    }
 }
 
 /**
  * pose() poses the animation on the specified frame number and changes the 
  * collider to the animation and collider having the specified anim_name.
  */
 template <typename T>
 inline void CollisionActor<T>::pose( const std::string& anim_name, const int frame )
 {
    // stop all animations and detach all colliders
    CollisionActor<T>::stop();
    
    // change the animation
    Actor<T>::get_anims().pose( anim_name, frame );
    
    // change the collider
    CollisionObject<T>::reparent_collider_to( anim_name, *this );
 }
 
 /**
  * stop() stops all animations and detaches all colliders.
  */
 template <typename T>
 inline void CollisionActor<T>::stop()
 {
    // stop the animations
    Actor<T>::get_anims().stop_all();
    
    // detach the colliders
    CollisionObject<T>::detach_colliders();
 }
 
 /**
  * stop() stops the animation and detaches the collider having the 
  * specified anim_name.
  *
  * @param (const std::string&) anim_name
  */
 template <typename T>
 inline void CollisionActor<T>::stop( const std::string& anim_name )
 {
    // stop the animation
    Actor<T>::get_anims().stop( anim_name );
    
    // detach the collider
    CollisionObject<T>::detach_collider( anim_name );
 }
 
 /**
  * turn_on_collision() makes this object capable of colliding.
  */
 template <typename T>
 inline void CollisionActor<T>::turn_on_collision()
 {
    AnimControlCollection& anim_collection( Actor<T>::get_anims() );
    size_t num_of_anims( anim_collection.get_num_anims() );
    //ColliderCollection& colliders( CollisionObject<T>::get_colliders() );
    
    for ( size_t i( 0 ); i < num_of_anims; ++i )
    {
        AnimControl* anim_Ptr( anim_collection.get_anim( i ) );
        
        // turn on the first collider found that matches the first active animation found associated with a collider
        if ( anim_Ptr->is_playing() )
        {
            CollisionHandlerEventNode<T>* collider_Ptr( CollisionObject<T>::get_collider( anim_Ptr->get_name() ) );
            
            if ( collider_Ptr != NULL )
            {
                collider_Ptr->reparent_to( *this );
                return;
            }
        }
    }
    
    /*CollisionHandlerEventNode<T>* collider_Ptr( NULL );
    
    std::string collider_key( _anim_collection.which_anim_playing() );
    if ( collider_key == "" )
    {
        collider_key = get_default_collider_name();
        collider_Ptr = _colliders.get( collider_key );
    }
    else
    {
        // turn on the first collider found that is currently active
        int index( collider_key.find_first_of( " " ) );
        if ( index != -1 )
        {
            std::string temp_key = collider_key.substr( 0, index );
            collider_Ptr = _colliders.get( temp_key );
            while ( collider_Ptr == NULL && index != -1 )
            {
                index = collider_key.find_first_of( " ", index + 1 );
                if ( index != -1 )
                {
                    temp_key = collider_key.substr( 0, index );
                    collider_Ptr = _colliders.get( temp_key );
                }
                else // no more spaces exist // no matching collider is found
                {
                    collider_Ptr = _colliders.get( collider_key );
                    
                    if ( collider_Ptr == NULL )
                        collider_Ptr = _colliders.get( get_default_collider_name() );
                    
                    break;
                }
            }
        }
        else
            collider_Ptr = _colliders.get( collider_key );
    }
    
    if ( collider_Ptr != NULL )
        collider_Ptr->reparent_to( *this );*/
 }
 
 /**
  * turn_off_collision() makes this object incapable of colliding.
  */
 template <typename T>
 inline void CollisionActor<T>::turn_off_collision()
 {
    //CollisionObject<T>::_colliders.detach_colliders();
    CollisionHandlerEventNode<T>* collider_Ptr( CollisionObject<T>::get_first_active_collider_found() );
    
    if ( collider_Ptr != NULL )
        collider_Ptr->detach_node();
    
    // turn off the collider matching the currently playing animation for this object
    //_colliders.detach_collider( _anim_collection.which_anim_playing() );
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /** FRIEND FUNCTIONS **/
 
 #endif // COLLISION_ACTOR_H